from ultralytics import YOLO
import cv2
import os
import glob

model = YOLO(os.path.join('Models','model.pt'))

class YOLOv8deploy:
    
    def __init__(self, model: YOLO, images_folder: str):
        
        """A simple class for deploying YOLOv8 model for detecting trading patterns in chart images

        Args:
            model (YOLO): YOLO model object
            images_folder (str): A path where images will be imported from
        """
        
        self.model = model
        self.images_folder = images_folder
        
    def _get_images(self, folder: str, img_extensions: list=['*.png', '*.jpg', '*.jpeg']) -> tuple: 
        
        """ A function to help us detect the number of images present in a folder

        Args:
            folder (str): A path where images are located
            img_extensions (list, optional): Image filenames extensions. Defaults to ['*.png', '*.jpg', '*.jpeg'].

        Returns:
            tuple: Returns the number of images present in a folder and their names
        """
        
        image_files = []
        for ext in img_extensions:
            image_files.extend(glob.glob(os.path.join(folder, ext)))

        return (len(image_files), image_files) # Get the number of images and their names
        
    def predict_image(self, img_name: str, hist: bool=True):
        
        """This function predicts a single image

        Args:
            img_name (str): name of the image
            hist (bool, optional): When set to false it means the function isn't predicting multiple instances and the outcome will be displayed.
            
            Defaults to True.
        """
        
        if os.path.exists(img_name) == False: # Check if an image exists
            print(f"Failed to detect patterns, {img_name} not found")
            return
        
        results = self.model.predict(source=img_name, save=True) # Predict an image 

        # Loop through the results
        for result in results:
            boxes = result.boxes  # Contains bounding boxes and confidence
            names = result.names  # Class index to name mapping

            if boxes is not None and len(boxes) > 0:
                for box in boxes:
                    cls_id = int(box.cls[0])  # class id
                    conf = box.conf[0].item()  # confidence score
                    label = names[cls_id]
                    
                    print(f"Detected: {label} (confidence: {conf:.2f})")
                    
                    # Open the saved image if this is a single (non-historical) run
                    
                if not hist:
                    base_name = os.path.splitext(os.path.basename(img_name))[0] + ".jpg"
                    saved_path = os.path.join(result.save_dir, base_name)
                    
                    print("saved path: ",saved_path)
                    
                    if os.path.exists(saved_path):
                        print(f"Opening detected image: {saved_path}")
                        img = cv2.imread(saved_path)
                        cv2.imshow("Detected Patterns", img)
                        cv2.waitKey(0)
                        cv2.destroyAllWindows()
            else:
                print("No detections.")


    def predict_images(self):
        
        _, image_names = self._get_images(self.images_folder) # Get all images from a folder
        
        for image_name in image_names:
            self.predict_image(img_name=image_name)
    
    
#############################################

images_path = r"C:\Users\Omega Joctan\AppData\Roaming\MetaQuotes\Terminal\F4F6C6D7A7155578A6DEA66D12B1D40D\MQL5\Files\Screenshots" # Change this for to the right path on your pc :)
symbol = "EURUSD"
timeframe = "PERIOD_H1"
imgs = 100

pattern_detector = YOLOv8deploy(model=model, images_folder=images_path)

pattern_detector.predict_images()


# pattern_detector.predict_image(img_name=os.path.join(images_path, f"{symbol}.{timeframe}.{11}.png"), 
#                                hist=False)
